<?php
session_start();
include 'config.php';

// ✅ Only allow admin to access
if(!isset($_SESSION['admin_id'])){
    header("Location: login.php");
    exit;
}

// ✅ Handle status update
if(isset($_GET['update_id']) && isset($_GET['status'])){
    $order_id = (int)$_GET['update_id'];
    $status = $_GET['status']; // Use DB enum values directly

    // Allow only valid statuses
    $valid_statuses = ['Pending', 'Confirmed', 'Shipped', 'Delivered', 'Cancelled'];
    if(in_array($status, $valid_statuses)){
        $stmt = $conn->prepare("UPDATE orders SET status=? WHERE order_id=?");
        $stmt->bind_param("si", $status, $order_id);
        $stmt->execute();
        $stmt->close();
    }

    header("Location: admin_orders.php");
    exit;
}

// ✅ Fetch all orders
$result = $conn->query("SELECT * FROM orders ORDER BY created_at DESC");
?>

<!DOCTYPE html>
<html>
<head>
    <title>Admin Orders</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">

<div class="container py-5">
    <h2 class="mb-4">Manage Orders</h2>

    <table class="table table-bordered table-striped align-middle">
        <thead class="table-dark">
            <tr>
                <th>ID</th>
                <th>User ID</th>
                <th>Delivery Name</th>
                <th>Phone</th>
                <th>Address</th>
                <th>City</th>
                <th>Pincode</th>
                <th>Total Amount</th>
                <th>Status</th>
                <th>Created At</th>
                <th>Payment</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
        <?php while($order = $result->fetch_assoc()): ?>
            <?php 
                $status = $order['status']; 
                $badgeClass = "bg-secondary";
                if($status == "Pending") $badgeClass = "bg-warning text-dark";
                if($status == "Confirmed") $badgeClass = "bg-success";
                if($status == "Shipped") $badgeClass = "bg-info text-dark";
                if($status == "Delivered") $badgeClass = "bg-primary";
                if($status == "Cancelled") $badgeClass = "bg-danger";
            ?>
            <tr>
                <td><?= $order['order_id'] ?></td>
                <td><?= $order['user_id'] ?></td>
                <td><?= htmlspecialchars($order['delivery_name']) ?></td>
                <td><?= htmlspecialchars($order['delivery_phone']) ?></td>
                <td><?= htmlspecialchars($order['delivery_address']) ?></td>
                <td><?= htmlspecialchars($order['delivery_city']) ?></td>
                <td><?= htmlspecialchars($order['delivery_pincode']) ?></td>
                <td>₹<?= number_format($order['total_amount'], 2) ?></td>
                <td>
                    <span class="badge <?= $badgeClass ?>"><?= $status ?></span>
                </td>
                <td><?= $order['created_at'] ?></td>
                <td><?= $order['payment_method'] ?></td>
                <td>
                    <?php if($status == "Pending"): ?>
                        <a href="admin_orders.php?update_id=<?= $order['order_id'] ?>&status=Confirmed" class="btn btn-success btn-sm">Confirm</a>
                        <a href="admin_orders.php?update_id=<?= $order['order_id'] ?>&status=Cancelled" class="btn btn-danger btn-sm">Reject</a>
                    <?php elseif($status == "Confirmed"): ?>
                        <a href="admin_orders.php?update_id=<?= $order['order_id'] ?>&status=Shipped" class="btn btn-info btn-sm">Ship</a>
                    <?php elseif($status == "Shipped"): ?>
                        <a href="admin_orders.php?update_id=<?= $order['order_id'] ?>&status=Delivered" class="btn btn-primary btn-sm">Deliver</a>
                    <?php else: ?>
                        <span class="text-muted">No action</span>
                    <?php endif; ?>
                </td>
            </tr>
        <?php endwhile; ?>
        </tbody>
    </table>
</div>

</body>
</html>
